from flask import Flask, request, render_template, send_file
import os
import requests
import xml.etree.ElementTree as ET
from datetime import datetime

app = Flask(__name__)

UPLOAD_FOLDER = 'generated_yml'  # Папка для сохранения YML-файлов
if not os.path.exists(UPLOAD_FOLDER):
    os.makedirs(UPLOAD_FOLDER)

# Функция для получения всех товаров из YML-файла
def get_all_items(yml_url):
    response = requests.get(yml_url)
    if response.status_code != 200:
        return None, []

    # Парсим YML-файл
    root = ET.fromstring(response.content)

    # Получаем шапку и конец файла
    shop_info = root.find('shop')
    offers = shop_info.find('offers').findall('offer')
    return root, offers

# Функция для создания нового YML-файла
def create_yml_part(part_number, root, offers, supplier_name):
    # Генерируем уникальное имя файла на основе времени и названия поставщика
    timestamp = datetime.now().strftime('%Y%m%d%H%M%S')
    filename = f'{supplier_name}_part_{part_number}_{timestamp}.xml'

    # Создаем новый XML-документ с шапкой
    new_root = ET.Element('yml_catalog', attrib={'date': root.attrib['date']})
    shop = ET.SubElement(new_root, 'shop')
    shop_info = root.find('shop')
    
    # Копируем основные элементы (например, название магазина, категории и т.д.)
    for element in list(shop_info):
        if element.tag != 'offers':  # Пропускаем блок товаров
            shop.append(element)
    
    # Добавляем товары
    offers_element = ET.SubElement(shop, 'offers')
    for offer in offers:
        offers_element.append(offer)

    # Сохраняем новый файл
    file_path = os.path.join(UPLOAD_FOLDER, filename)
    tree = ET.ElementTree(new_root)
    tree.write(file_path, encoding='utf-8', xml_declaration=True)
    
    return filename

# Отображение формы
@app.route('/')
def index():
    return render_template('index.html')  # Отображаем HTML-форму

# Обработка формы
@app.route('/process', methods=['POST'])
def process():
    url = request.form['url']  # Получаем URL из формы
    count = int(request.form['count'])  # Получаем количество товаров
    supplier_name = request.form['supplier_name']  # Получаем название поставщика (можно использовать для уникальности)

    # Получаем все товары и шапку YML-файла
    root, offers = get_all_items(url)

    total_items = len(offers)

    if total_items == 0:
        return "Не удалось найти товары в YML-файле."

    # Подсчитываем, сколько частей нам нужно
    num_parts = (total_items + count - 1) // count  # Число частей
    parts = []

    for i in range(num_parts):
        # Разделяем товары на части
        part_offers = offers[i * count:(i + 1) * count]
        
        # Создаем YML-файл для этой части
        filename = create_yml_part(i + 1, root, part_offers, supplier_name)
        
        # Генерируем ссылку для скачивания файла
        part_url = f'/download/{filename}'
        parts.append(part_url)

    # Возвращаем ссылки на скачивание файлов
    return '<br>'.join([f'Часть {i + 1}: <a href="{part}" target="_blank">{part}</a>' for i, part in enumerate(parts)])

# Обработка скачивания файлов
@app.route('/download/<filename>')
def download_file(filename):
    file_path = os.path.join(UPLOAD_FOLDER, filename)
    if os.path.exists(file_path):
        return send_file(file_path)
    return "Файл не найден."

if __name__ == '__main__':
    app.run(debug=True)
